<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * This page prints a review of a particular quiz attempt
 *
 * It is used either by the student whose attempts this is, after the attempt,
 * or by a teacher reviewing another's attempt during or afterwards.
 *
 * @package   mod_quiz
 * @copyright 1999 onwards Martin Dougiamas  {@link http://moodle.com}
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */


require_once(dirname(__FILE__) . '/../../config.php');
require_once($CFG->dirroot . '/mod/quiz/locallib.php');
require_once($CFG->dirroot . '/mod/quiz/report/reportlib.php');

$attemptid = required_param('attempt', PARAM_INT);
$page      = optional_param('page', 0, PARAM_INT);
$showall   = optional_param('showall', null, PARAM_BOOL);

$url = new moodle_url('/mod/quiz/review.php', array('attempt'=>$attemptid));
if ($page !== 0) {
    $url->param('page', $page);
} else if ($showall) {
    $url->param('showall', $showall);
}
$PAGE->set_url($url);

$attemptobj = quiz_attempt::create($attemptid);
$page = $attemptobj->force_page_number_into_range($page);

// Now we can validate the params better, re-genrate the page URL.
if ($showall === null) {
    $showall = $page == 0 && $attemptobj->get_default_show_all('review');
}
$PAGE->set_url($attemptobj->review_url(null, $page, $showall));

// Check login.
require_login($attemptobj->get_course(), false, $attemptobj->get_cm());
$attemptobj->check_review_capability();

// Create an object to manage all the other (non-roles) access rules.
$accessmanager = $attemptobj->get_access_manager(time());
$accessmanager->setup_attempt_page($PAGE);

$options = $attemptobj->get_display_options(true);

// Check permissions.
if ($attemptobj->is_own_attempt()) {
    if (!$attemptobj->is_finished()) {
        redirect($attemptobj->attempt_url(null, $page));

    } else if (!$options->attempt) {
        $accessmanager->back_to_view_page($PAGE->get_renderer('mod_quiz'),
                $attemptobj->cannot_review_message());
    }

} else if (!$attemptobj->is_review_allowed()) {
    throw new moodle_quiz_exception($attemptobj->get_quizobj(), 'noreviewattempt');
}

// Load the questions and states needed by this page.
if ($showall) {
    $questionids = $attemptobj->get_slots();
} else {
    $questionids = $attemptobj->get_slots($page);
}

// Save the flag states, if they are being changed.
if ($options->flags == question_display_options::EDITABLE && optional_param('savingflags', false,
        PARAM_BOOL)) {
    require_sesskey();
    $attemptobj->save_question_flags();
    redirect($attemptobj->review_url(null, $page, $showall));
}

// Work out appropriate title and whether blocks should be shown.
if ($attemptobj->is_preview_user() && $attemptobj->is_own_attempt()) {
    $strreviewtitle = get_string('reviewofpreview', 'quiz');
    navigation_node::override_active_url($attemptobj->start_attempt_url());

} else {
    $strreviewtitle = get_string('reviewofattempt', 'quiz', $attemptobj->get_attempt_number());
    if (empty($attemptobj->get_quiz()->showblocks) && !$attemptobj->is_preview_user()) {
        $PAGE->blocks->show_only_fake_blocks();
    }
}

// Set up the page header.
$headtags = $attemptobj->get_html_head_contributions($page, $showall);
$PAGE->set_title($attemptobj->get_quiz_name());
$PAGE->set_heading($attemptobj->get_course()->fullname);

// Summary table start. ============================================================================

// Work out some time-related things.
$attempt = $attemptobj->get_attempt();
$quiz = $attemptobj->get_quiz();
$overtime = 0;

if ($attempt->state == quiz_attempt::FINISHED) {
    if ($timetaken = ($attempt->timefinish - $attempt->timestart)) {
        if ($quiz->timelimit && $timetaken > ($quiz->timelimit + 60)) {
            $overtime = $timetaken - $quiz->timelimit;
            $overtime = format_time($overtime);
        }
        $timetaken = format_time($timetaken);
    } else {
        $timetaken = "-";
    }
} else {
    $timetaken = get_string('unfinished', 'quiz');
}

// Prepare summary informat about the whole attempt.
$summarydata = array();
if (!$attemptobj->get_quiz()->showuserpicture && $attemptobj->get_userid() != $USER->id) {
    // If showuserpicture is true, the picture is shown elsewhere, so don't repeat it.
    $student = $DB->get_record('user', array('id' => $attemptobj->get_userid()));
    $usrepicture = new user_picture($student);
    $usrepicture->courseid = $attemptobj->get_courseid();
    $summarydata['user'] = array(
        'title'   => $usrepicture,
        'content' => new action_link(new moodle_url('/user/view.php', array(
                                'id' => $student->id, 'course' => $attemptobj->get_courseid())),
                          fullname($student, true)),
    );
}

if ($attemptobj->has_capability('mod/quiz:viewreports')) {
    $attemptlist = $attemptobj->links_to_other_attempts($attemptobj->review_url(null, $page,
            $showall));
    if ($attemptlist) {
        $summarydata['attemptlist'] = array(
            'title'   => get_string('attempts', 'quiz'),
            'content' => $attemptlist,
        );
    }
}

// Timing information.
$summarydata['startedon'] = array(
    'title'   => get_string('startedon', 'quiz'),
    'content' => userdate($attempt->timestart),
);

$summarydata['state'] = array(
    'title'   => get_string('attemptstate', 'quiz'),
    'content' => quiz_attempt::state_name($attempt->state),
);

if ($attempt->state == quiz_attempt::FINISHED) {
    $summarydata['completedon'] = array(
        'title'   => get_string('completedon', 'quiz'),
        'content' => userdate($attempt->timefinish),
    );
    $summarydata['timetaken'] = array(
        'title'   => get_string('timetaken', 'quiz'),
        'content' => $timetaken,
    );
}

if (!empty($overtime)) {
    $summarydata['overdue'] = array(
        'title'   => get_string('overdue', 'quiz'),
        'content' => $overtime,
    );
}

// Show marks (if the user is allowed to see marks at the moment).
$grade = quiz_rescale_grade($attempt->sumgrades, $quiz, false);
if ($options->marks >= question_display_options::MARK_AND_MAX && quiz_has_grades($quiz)) {

    if ($attempt->state != quiz_attempt::FINISHED) {
        // Cannot display grade.

    } else if (is_null($grade)) {
        $summarydata['grade'] = array(
            'title'   => get_string('grade', 'quiz'),
            'content' => quiz_format_grade($quiz, $grade),
        );

    } else {
        // Show raw marks only if they are different from the grade (like on the view page).
        if ($quiz->grade != $quiz->sumgrades) {
            $a = new stdClass();
            $a->grade = quiz_format_grade($quiz, $attempt->sumgrades);
            $a->maxgrade = quiz_format_grade($quiz, $quiz->sumgrades);
            $summarydata['marks'] = array(
                'title'   => get_string('marks', 'quiz'),
                'content' => get_string('outofshort', 'quiz', $a),
            );
        }

        // Now the scaled grade.
        $a = new stdClass();
        $a->grade = html_writer::tag('b', quiz_format_grade($quiz, $grade));
        $a->maxgrade = quiz_format_grade($quiz, $quiz->grade);
        if ($quiz->grade != 100) {
            $a->percent = html_writer::tag('b', format_float(
                    $attempt->sumgrades * 100 / $quiz->sumgrades, 0));
            $formattedgrade = get_string('outofpercent', 'quiz', $a);
        } else {
            $formattedgrade = get_string('outof', 'quiz', $a);
        }
        $summarydata['grade'] = array(
            'title'   => get_string('grade', 'quiz'),
            'content' => $formattedgrade,
        );
    }
}

// Any additional summary data from the behaviour.
$summarydata = array_merge($summarydata, $attemptobj->get_additional_summary_data($options));



//added by taz for email sending when a user attempt quiz
require_once($CFG->dirroot.'/user/profile/lib.php');

profile_load_data($USER);


// if($attempt->certificate == 1 && $attempt->percent >= 80){
// redirect($CFG->wwwroot.'/course/view.php?id='.$attemptobj->get_courseid(),'You have already passed this test - congratulations!');
// }

/*

$percents = format_float($attempt->sumgrades * 100 / $quiz->sumgrades, 0);
if($attempt->state == quiz_attempt::FINISHED){
if($attempt->certificate == 0){
if($percents >= 79){
date_default_timezone_set("America/New_York");
$parameters['mvUserid'] = '2Babydrive';
$parameters['mvPassword'] = '3Orange$';
$parameters['mvSchoolid'] = '22847';
$parameters['mvClassDate'] = date('mdY',$attempt->timefinish);
$parameters['mvStartTime'] = date('Hi',$attempt->timefinish);
$parameters['mvSchoolIns'] = '60237';
$parameters['mvSchoolCourse'] = '30749';
$parameters['mvFirstName'] = $USER->firstname;
$parameters['mvLastName'] = $USER->lastname;
$parameters['mvDob'] = date('mdY',$USER->profile['dateofbirth']);
if($USER->profile['Gender'] == 'Male'){
$gender = 'M';
}else{
$gender = 'F';
}
if($USER->profile['reasonforattending'] == "School Election"){
$reasonforattending = 'B1';
} elseif ($USER->profile['reasonforattendance'] == "Court Order"){
$reasonforattendance = 'B3';
} elseif ($USER->profile['reasonforattendance'] == "Insurance"){
$reasonforattendance = 'B4';
}else{
$reasonforattendance = 'B2';
}
if($USER->profile['stateofrecord'] == "AK"){ $stateofrecord = 'AK';
} elseif ($USER->profile['stateofrecord'] == "AL"){ $stateofrecord = 'AL';
} elseif ($USER->profile['stateofrecord'] == "AR"){ $stateofrecord = 'AR';
} elseif ($USER->profile['stateofrecord'] == "AZ"){ $stateofrecord = 'AZ';
} elseif ($USER->profile['stateofrecord'] == "CA"){ $stateofrecord = 'CA';
} elseif ($USER->profile['stateofrecord'] == "CO"){ $stateofrecord = 'CO';
} elseif ($USER->profile['stateofrecord'] == "CT"){ $stateofrecord = 'CT';
} elseif ($USER->profile['stateofrecord'] == "DC"){ $stateofrecord = 'DC';
} elseif ($USER->profile['stateofrecord'] == "DE"){ $stateofrecord = 'DE';
} elseif ($USER->profile['stateofrecord'] == "FL"){ $stateofrecord = 'FL';
} elseif ($USER->profile['stateofrecord'] == "GA"){ $stateofrecord = 'GA';
} elseif ($USER->profile['stateofrecord'] == "HI"){ $stateofrecord = 'HI';
} elseif ($USER->profile['stateofrecord'] == "IA"){ $stateofrecord = 'IA';
} elseif ($USER->profile['stateofrecord'] == "ID"){ $stateofrecord = 'ID';
} elseif ($USER->profile['stateofrecord'] == "IL"){ $stateofrecord = 'IL';
} elseif ($USER->profile['stateofrecord'] == "IN"){ $stateofrecord = 'IN';
} elseif ($USER->profile['stateofrecord'] == "KS"){ $stateofrecord = 'KS';
} elseif ($USER->profile['stateofrecord'] == "KY"){ $stateofrecord = 'KY';
} elseif ($USER->profile['stateofrecord'] == "LA"){ $stateofrecord = 'LA';
} elseif ($USER->profile['stateofrecord'] == "MA"){ $stateofrecord = 'MA';
} elseif ($USER->profile['stateofrecord'] == "MD"){ $stateofrecord = 'MD';
} elseif ($USER->profile['stateofrecord'] == "ME"){ $stateofrecord = 'ME';
} elseif ($USER->profile['stateofrecord'] == "MI"){ $stateofrecord = 'MI';
} elseif ($USER->profile['stateofrecord'] == "MN"){ $stateofrecord = 'MN';
} elseif ($USER->profile['stateofrecord'] == "MO"){ $stateofrecord = 'MO';
} elseif ($USER->profile['stateofrecord'] == "MS"){ $stateofrecord = 'MS';
} elseif ($USER->profile['stateofrecord'] == "MT"){ $stateofrecord = 'MT';
} elseif ($USER->profile['stateofrecord'] == "NC"){ $stateofrecord = 'NC';
} elseif ($USER->profile['stateofrecord'] == "ND"){ $stateofrecord = 'ND';
} elseif ($USER->profile['stateofrecord'] == "NE"){ $stateofrecord = 'NE';
} elseif ($USER->profile['stateofrecord'] == "NH"){ $stateofrecord = 'NH';
} elseif ($USER->profile['stateofrecord'] == "NJ"){ $stateofrecord = 'NJ';
} elseif ($USER->profile['stateofrecord'] == "NM"){ $stateofrecord = 'NM';
} elseif ($USER->profile['stateofrecord'] == "NV"){ $stateofrecord = 'NV';
} elseif ($USER->profile['stateofrecord'] == "NY"){ $stateofrecord = 'NY';
} elseif ($USER->profile['stateofrecord'] == "OH"){ $stateofrecord = 'OH';
} elseif ($USER->profile['stateofrecord'] == "OK"){ $stateofrecord = 'OK';
} elseif ($USER->profile['stateofrecord'] == "OR"){ $stateofrecord = 'OR';
} elseif ($USER->profile['stateofrecord'] == "PA"){ $stateofrecord = 'PA';
} elseif ($USER->profile['stateofrecord'] == "RI"){ $stateofrecord = 'RI';
} elseif ($USER->profile['stateofrecord'] == "SC"){ $stateofrecord = 'SC';
} elseif ($USER->profile['stateofrecord'] == "SD"){ $stateofrecord = 'SD';
} elseif ($USER->profile['stateofrecord'] == "TN"){ $stateofrecord = 'TN';
} elseif ($USER->profile['stateofrecord'] == "TX"){ $stateofrecord = 'TX';
} elseif ($USER->profile['stateofrecord'] == "UT"){ $stateofrecord = 'UT';
} elseif ($USER->profile['stateofrecord'] == "VA"){ $stateofrecord = 'VA';
} elseif ($USER->profile['stateofrecord'] == "VT"){ $stateofrecord = 'VT';
} elseif ($USER->profile['stateofrecord'] == "WA"){ $stateofrecord = 'WA';
} elseif ($USER->profile['stateofrecord'] == "WI"){ $stateofrecord = 'WI';
} elseif ($USER->profile['stateofrecord'] == "WV"){ $stateofrecord = 'WV';
} elseif ($USER->profile['stateofrecord'] == "WY"){ $stateofrecord = 'WY';
} elseif ($USER->profile['stateofrecord'] == "AA"){ $stateofrecord = 'AA';
} elseif ($USER->profile['stateofrecord'] == "AB"){ $stateofrecord = 'AB';
} elseif ($USER->profile['stateofrecord'] == "AE"){ $stateofrecord = 'AE';
} elseif ($USER->profile['stateofrecord'] == "AP"){ $stateofrecord = 'AP';
} elseif ($USER->profile['stateofrecord'] == "AS"){ $stateofrecord = 'AS';
} elseif ($USER->profile['stateofrecord'] == "BC"){ $stateofrecord = 'BC';
} elseif ($USER->profile['stateofrecord'] == "FM"){ $stateofrecord = 'FM';
} elseif ($USER->profile['stateofrecord'] == "GU"){ $stateofrecord = 'GU';
} elseif ($USER->profile['stateofrecord'] == "MB"){ $stateofrecord = 'MB';
} elseif ($USER->profile['stateofrecord'] == "MH"){ $stateofrecord = 'MH';
} elseif ($USER->profile['stateofrecord'] == "MP"){ $stateofrecord = 'MP';
} elseif ($USER->profile['stateofrecord'] == "NB"){ $stateofrecord = 'NB';
} elseif ($USER->profile['stateofrecord'] == "NF"){ $stateofrecord = 'NF';
} elseif ($USER->profile['stateofrecord'] == "NS"){ $stateofrecord = 'NS';
} elseif ($USER->profile['stateofrecord'] == "NT"){ $stateofrecord = 'NT';
} elseif ($USER->profile['stateofrecord'] == "ON"){ $stateofrecord = 'ON';
} elseif ($USER->profile['stateofrecord'] == "PE"){ $stateofrecord = 'PE';
} elseif ($USER->profile['stateofrecord'] == "PQ"){ $stateofrecord = 'PQ';
} elseif ($USER->profile['stateofrecord'] == "PR"){ $stateofrecord = 'PR';
} elseif ($USER->profile['stateofrecord'] == "PW"){ $stateofrecord = 'PW';
} elseif ($USER->profile['stateofrecord'] == "SK"){ $stateofrecord = 'SK';
} else { $stateofrecord = 'VI';
}
if($USER->profile['ZipCode'] == "33010"){ $ZipCode= '33010';
} elseif ($USER->profile['ZipCode'] == "33012"){ $ZipCode= '33012';
} elseif ($USER->profile['ZipCode'] == "33013"){ $ZipCode= '33013';
} elseif ($USER->profile['ZipCode'] == "33014"){ $ZipCode= '33014';
} elseif ($USER->profile['ZipCode'] == "33015"){ $ZipCode= '33015';
} elseif ($USER->profile['ZipCode'] == "33016"){ $ZipCode= '33016';
} elseif ($USER->profile['ZipCode'] == "33018"){ $ZipCode= '33018';
} elseif ($USER->profile['ZipCode'] == "33030"){ $ZipCode= '33030';
} elseif ($USER->profile['ZipCode'] == "33031"){ $ZipCode= '33031';
} elseif ($USER->profile['ZipCode'] == "33032"){ $ZipCode= '33032';
} elseif ($USER->profile['ZipCode'] == "33033"){ $ZipCode= '33033';
} elseif ($USER->profile['ZipCode'] == "33034"){ $ZipCode= '33034';
} elseif ($USER->profile['ZipCode'] == "33054"){ $ZipCode= '33054';
} elseif ($USER->profile['ZipCode'] == "33056"){ $ZipCode= '33056';
} elseif ($USER->profile['ZipCode'] == "33122"){ $ZipCode= '33122';
} elseif ($USER->profile['ZipCode'] == "33125"){ $ZipCode= '33125';
} elseif ($USER->profile['ZipCode'] == "33126"){ $ZipCode= '33126';
} elseif ($USER->profile['ZipCode'] == "33127"){ $ZipCode= '33127';
} elseif ($USER->profile['ZipCode'] == "33128"){ $ZipCode= '33128';
} elseif ($USER->profile['ZipCode'] == "33129"){ $ZipCode= '33129';
} elseif ($USER->profile['ZipCode'] == "33130"){ $ZipCode= '33130';
} elseif ($USER->profile['ZipCode'] == "33131"){ $ZipCode= '33131';
} elseif ($USER->profile['ZipCode'] == "33132"){ $ZipCode= '33132';
} elseif ($USER->profile['ZipCode'] == "33133"){ $ZipCode= '33133';
} elseif ($USER->profile['ZipCode'] == "33134"){ $ZipCode= '33134';
} elseif ($USER->profile['ZipCode'] == "33135"){ $ZipCode= '33135';
} elseif ($USER->profile['ZipCode'] == "33136"){ $ZipCode= '33136';
} elseif ($USER->profile['ZipCode'] == "33137"){ $ZipCode= '33137';
} elseif ($USER->profile['ZipCode'] == "33138"){ $ZipCode= '33138';
} elseif ($USER->profile['ZipCode'] == "33139"){ $ZipCode= '33139';
} elseif ($USER->profile['ZipCode'] == "33140"){ $ZipCode= '33140';
} elseif ($USER->profile['ZipCode'] == "33141"){ $ZipCode= '33141';
} elseif ($USER->profile['ZipCode'] == "33142"){ $ZipCode= '33142';
} elseif ($USER->profile['ZipCode'] == "33143"){ $ZipCode= '33143';
} elseif ($USER->profile['ZipCode'] == "33144"){ $ZipCode= '33144';
} elseif ($USER->profile['ZipCode'] == "33145"){ $ZipCode= '33145';
} elseif ($USER->profile['ZipCode'] == "33146"){ $ZipCode= '33146';
} elseif ($USER->profile['ZipCode'] == "33147"){ $ZipCode= '33147';
} elseif ($USER->profile['ZipCode'] == "33149"){ $ZipCode= '44149';
} elseif ($USER->profile['ZipCode'] == "33150"){ $ZipCode= '33150';
} elseif ($USER->profile['ZipCode'] == "33154"){ $ZipCode= '33154';
} elseif ($USER->profile['ZipCode'] == "33155"){ $ZipCode= '33155';
} elseif ($USER->profile['ZipCode'] == "33156"){ $ZipCode= '33156';
} elseif ($USER->profile['ZipCode'] == "33157"){ $ZipCode= '33157';
} elseif ($USER->profile['ZipCode'] == "33158"){ $ZipCode= '33158';
} elseif ($USER->profile['ZipCode'] == "33160"){ $ZipCode= '33160';
} elseif ($USER->profile['ZipCode'] == "33161"){ $ZipCode= '33161';
} elseif ($USER->profile['ZipCode'] == "33162"){ $ZipCode= '33162';
} elseif ($USER->profile['ZipCode'] == "33165"){ $ZipCode= '33165';
} elseif ($USER->profile['ZipCode'] == "33166"){ $ZipCode= '33166';
} elseif ($USER->profile['ZipCode'] == "33167"){ $ZipCode= '33167';
} elseif ($USER->profile['ZipCode'] == "33168"){ $ZipCode= '33168';
} elseif ($USER->profile['ZipCode'] == "33169"){ $ZipCode= '33169';
} elseif ($USER->profile['ZipCode'] == "33170"){ $ZipCode= '33170';
} elseif ($USER->profile['ZipCode'] == "33172"){ $ZipCode= '33172';
} elseif ($USER->profile['ZipCode'] == "33173"){ $ZipCode= '33173';
} elseif ($USER->profile['ZipCode'] == "33174"){ $ZipCode= '33174';
} elseif ($USER->profile['ZipCode'] == "33175"){ $ZipCode= '33175';
} elseif ($USER->profile['ZipCode'] == "33176"){ $ZipCode= '33176';
} elseif ($USER->profile['ZipCode'] == "33177"){ $ZipCode= '33177';
} elseif ($USER->profile['ZipCode'] == "33178"){ $ZipCode= '33178';
} elseif ($USER->profile['ZipCode'] == "33179"){ $ZipCode= '33179';
} elseif ($USER->profile['ZipCode'] == "33180"){ $ZipCode= '33180';
} elseif ($USER->profile['ZipCode'] == "33181"){ $ZipCode= '33181';
} elseif ($USER->profile['ZipCode'] == "33182"){ $ZipCode= '33182';
} elseif ($USER->profile['ZipCode'] == "33183"){ $ZipCode= '33183';
} elseif ($USER->profile['ZipCode'] == "33184"){ $ZipCode= '33184';
} elseif ($USER->profile['ZipCode'] == "33185"){ $ZipCode= '33185';
} elseif ($USER->profile['ZipCode'] == "33186"){ $ZipCode= '33186';
} elseif ($USER->profile['ZipCode'] == "33187"){ $ZipCode= '33187';
} elseif ($USER->profile['ZipCode'] == "33189"){ $ZipCode= '33189';
} elseif ($USER->profile['ZipCode'] == "33193"){ $ZipCode= '33193';
} else { $ZipCode= '33196';
}


$parameters['mvSex'] = $gender;
$parameters['mvSocialSN'] = $USER->profile_field_IDnumber;
$parameters['mvReasonAttending'] = $reasonforattending;
$parameters['mvDriversLicense'] = $USER->profile['DriversLicense'];
$parameters['mvCitationDate'] = date('mdY',$USER->profile['dateofcitation']);
$parameters['mvCitationCounty'] = 'Dade';
$parameters['mvCitationNumber'] = $USER->profile['CitationNumber'];
$parameters['mvdlStateOfRecordCode'] = $stateofrecord;
$parameters['mvZipCode'] = $USER->profile['ZipCode'];
$parameters['mvEmail'] = $USER->email; 

$client=new SoapClient("https://services.flhsmv.gov/DriverSchoolWebService/wsPrimerComponentService.svc?singleWsdl");  
$result = $client->wsVerifyData($parameters);
$msg = ''; 
$to = "$USER->email";
$subject = "Your Florida Confirmation Number has been sent to the DMV";
$msg ="Dear $USER->username\n" . "</br>" . "</br>" ;
/* $msg .=" your Test name is: $quiz->name" . "<br>" . "<br>" ; */
/*
$msg .= " Congratulations for passing Basic Driver Improvement! This confirmation number has been sent to the DMV. The DMV does not issue a physical certificate, but only a number." . "<br>" . "<br>";
$msg .= "Your confirmation number is $result->wsVerifyDataResult\n" . "<br>" . "<br>";
$msg .= "If the number above is BD000, that means you made an error when you created your profile. Email us at CustomerCare@15DollarFloridaBDI.com and we will take care of it.";
// $msg .= "Your certificate number is $result->wsVerifyDataResponse\n" . "<br>" . "<br>";
//$msg .= "Next step: Click on my link in the menu tab that says Permit Checklist, and it is all explained there. Basically, you will click on the //Driver's Manual, study it on your own, and then go to the DMV and take the Knowledge Exam, and they will guide you from there." . "<br>";
$headers = "MIME-Version: 1.0" . "\r\n";
$headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
$headers .= "From: $CFG->noreplyaddress" . "\r\n" ;
mail($to,$subject,$msg, $headers); 
$DB->set_field('quiz_attempts', 'certificate', 1, array('id' => $attemptid));
$DB->set_field('quiz_attempts', 'percent', $percents, array('id' => $attemptid));
}
}
}
/*
$parameters['mvCitationCounty'] = $USER->profile['County'];
if($USER->profile['County'] == "Alachua"){ $County = 'Alachua';
} elseif ($USER->profile['County'] == "Baker"){ $County = 'Baker';
} elseif ($USER->profile['County'] == "Bay"){ $County = 'Bay';
} elseif ($USER->profile['County'] == "Bradford"){ $County = 'Bradford';
} elseif ($USER->profile['County'] == "Brevard"){ $County = 'Brevard';
} elseif ($USER->profile['County'] == "Broward"){ $County = 'Broward';
} elseif ($USER->profile['County'] == "Calhoun"){ $County = 'Calhoun';
} elseif ($USER->profile['County'] == "Charlotte"){ $County = 'Charlotte';
} elseif ($USER->profile['County'] == "Citrus"){ $County = 'Citrus';
} elseif ($USER->profile['County'] == "Clay"){ $County = 'Clay';
} elseif ($USER->profile['County'] == "Collier"){ $County = 'Collier';
} elseif ($USER->profile['County'] == "Columbia"){ $County = 'Columbia';
} elseif ($USER->profile['County'] == "Dade"){ $County = 'Dade';
} elseif ($USER->profile['County'] == "De Soto"){ $County = 'De Soto';
} elseif ($USER->profile['County'] == "Dixie"){ $County = 'Dixie';
} elseif ($USER->profile['County'] == "Duval"){ $County = 'Duval';
} elseif ($USER->profile['County'] == "Escambia"){ $County = 'Escambia';
} elseif ($USER->profile['County'] == "Flagler"){ $County = 'Flagler';
} elseif ($USER->profile['County'] == "Franklin"){ $County = 'Franklin';
} elseif ($USER->profile['County'] == "Gadsden"){ $County = 'Gadsden';
} elseif ($USER->profile['County'] == "Gilchrist"){ $County = 'Gilchrist';
} elseif ($USER->profile['County'] == "Glades"){ $County = 'Glades';
} elseif ($USER->profile['County'] == "Gulf"){ $County = 'Gulf';
} elseif ($USER->profile['County'] == "Hamilton"){ $County = 'Hamilton';
} elseif ($USER->profile['County'] == "Hardee"){ $County = 'Hardee';
} elseif ($USER->profile['County'] == "Hendry"){ $County = 'Hendry';
} elseif ($USER->profile['County'] == "Hernando"){ $County = 'Hernando';
} elseif ($USER->profile['County'] == "Highland"){ $County = 'Highland';
} elseif ($USER->profile['County'] == "Hillsborough"){ $County = 'Hillsborough';
} elseif ($USER->profile['County'] == "Holmes"){ $County = 'Holmes';
} elseif ($USER->profile['County'] == "Indian River"){ $County = 'Indian River';
} elseif ($USER->profile['County'] == "Jackson"){ $County = 'Jackson';
} elseif ($USER->profile['County'] == "Jefferson"){ $County = 'Jefferson';
} elseif ($USER->profile['County'] == "Lafayette"){ $County = 'Lafayette';
} elseif ($USER->profile['County'] == "Lake"){ $County = 'Lake';
} elseif ($USER->profile['County'] == "Lee"){ $County = 'Lee';
} elseif ($USER->profile['County'] == "Leon"){ $County = 'Leon';
} elseif ($USER->profile['County'] == "Levy"){ $County = 'Levy';
} elseif ($USER->profile['County'] == "Liberty"){ $County = 'Liberty';
} elseif ($USER->profile['County'] == "Madison"){ $County = 'Madison';
} elseif ($USER->profile['County'] == "Manatee"){ $County = 'Manatee';
} elseif ($USER->profile['County'] == "Marion"){ $County = 'Marion';
} elseif ($USER->profile['County'] == "Martin"){ $County = 'Martin';
} elseif ($USER->profile['County'] == "Monroe"){ $County = 'Monroe';
} elseif ($USER->profile['County'] == "Nassau"){ $County = 'Nassau';
} elseif ($USER->profile['County'] == "Okaloosa"){ $County = 'Okaloosa';
} elseif ($USER->profile['County'] == "Okeechobee"){ $County = 'Okeechobee';
} elseif ($USER->profile['County'] == "Orange"){ $County = 'Orange';
} elseif ($USER->profile['County'] == "Osceola"){ $County = 'Osceola';
} elseif ($USER->profile['County'] == "Palm Beach"){ $County = 'Palm Beach';
} elseif ($USER->profile['County'] == "Pasco"){ $County = 'Pasco';
} elseif ($USER->profile['County'] == "Pinellas"){ $County = 'Pinellas';
} elseif ($USER->profile['County'] == "Polk"){ $County = 'Polk';
} elseif ($USER->profile['County'] == "Putnam"){ $County = 'Putnam';
} elseif ($USER->profile['County'] == "Santa Rosa"){ $County = 'Santa Rosa';
} elseif ($USER->profile['County'] == "Sarasota"){ $County = 'Sarasota';
} elseif ($USER->profile['County'] == "Seminole"){ $County = 'Seminole';
} elseif ($USER->profile['County'] == "St Johns"){ $County = 'St Johns';
} elseif ($USER->profile['County'] == "St Lucie"){ $County = 'St Lucie';
} elseif ($USER->profile['County'] == "Sumter"){ $County = 'Sumter';
} elseif ($USER->profile['County'] == "Suwannee"){ $County = 'Suwannee';
} elseif ($USER->profile['County'] == "Taylor"){ $County = 'Taylor';
} elseif ($USER->profile['County'] == "Union"){ $County = 'Union';
} elseif ($USER->profile['County'] == "Volusia"){ $County = 'Volusia';
} elseif ($USER->profile['County'] == "Wakulla"){ $County = 'Wakulla';
} elseif ($USER->profile['County'] == "Walton"){ $County = 'Walton';
} else { $County = 'Washington';
} */

*/
//end added by taz for email sending when a user attempt quiz



// Feedback if there is any, and the user is allowed to see it now.
$feedback = $attemptobj->get_overall_feedback($grade);
if ($options->overallfeedback && $feedback) {
    $summarydata['feedback'] = array(
        'title'   => get_string('feedback', 'quiz'),
        'content' => $feedback,
    );
}

// Summary table end. ==============================================================================

if ($showall) {
    $slots = $attemptobj->get_slots();
    $lastpage = true;
} else {
    $slots = $attemptobj->get_slots($page);
    $lastpage = $attemptobj->is_last_page($page);
}

$output = $PAGE->get_renderer('mod_quiz');

// Arrange for the navigation to be displayed.
$navbc = $attemptobj->get_navigation_panel($output, 'quiz_review_nav_panel', $page, $showall);
$regions = $PAGE->blocks->get_regions();
$PAGE->blocks->add_fake_block($navbc, reset($regions));

echo $output->review_page($attemptobj, $slots, $page, $showall, $lastpage, $options, $summarydata);

// Trigger an event for this review.
$params = array(
    'objectid' => $attemptobj->get_attemptid(),
    'relateduserid' => $attemptobj->get_userid(),
    'courseid' => $attemptobj->get_courseid(),
    'context' => context_module::instance($attemptobj->get_cmid()),
    'other' => array(
        'quizid' => $attemptobj->get_quizid()
    )
);
$event = \mod_quiz\event\attempt_reviewed::create($params);
$event->add_record_snapshot('quiz_attempts', $attemptobj->get_attempt());
$event->trigger();
